using System;
using System.Data;
using Method = System.Reflection.MethodBase;

using TABLE = gov.va.med.vbecs.Common.VbecsTables.PatientSpecimen;

namespace gov.va.med.vbecs.BOL
{

	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Cameron Taylor</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>1/16/2003</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>Specimen business object class</summary>

	#endregion

	public class Specimen:BaseBusinessObject
	{
		#region Constants
		/// <summary>
		/// Specimen Table
		/// </summary>
		public const string SPECIMENTABLE = "SpecimenTable";
		/// <summary>
		/// Specimen Comment Table
		/// </summary>
		public const string SPECIMENCOMMENTTABLE = "SpecimenCommentTable";
		/// <summary>
		/// Insert Specimen Comment Table
		/// </summary>
		public const string INSERTSPECIMENCOMMENTTABLE = "InsertSpecimenCommentTable";

//		/// <summary>
//		/// Default Expiration Days.
//		/// </summary>
//		public const short DEFAULT_EXPIRATION_DAYS = 3;
//		public const short MAXIMUM_SPECIMEN_EXPIRATION_DAYS = 90;

		/// <summary>
		/// To work around "broken rules" I'm setting a generic 2-for-1 deal.  This applies to the indicator and comment
		/// </summary>
		//private const string UNACCEPTABLE_SPECIMEN_INDICATOR = "UnacceptableSpecimenIndicator";

		#endregion

		#region Variables

		private BOL.CannedComment _unacceptableSpecimenComment;
		//private BOL.CannedComment _extendedSpecimenComment;
		private short _maximumSpecimenExpirationDays;
		private bool _isOverMaxDays;
		//private DataTable _testResults;
		private bool _previouslySavedAsUnacceptable;
		private bool _isAssociatedWithPendingOrder = false;
		

		/// <summary>
		/// _hasAntibodyIdentification
		/// </summary>
		private bool _hasAntibodyIdentification = false;

		/// <summary>
		/// _hasTransfusionHistory
		/// </summary>
		private bool _hasTransfusionHistory = false;

		/// <summary>
		/// _hasDiagnosticTests
		/// </summary>
		private bool _hasDiagnosticTests = false;

		/// <summary>
		/// _patient
		/// </summary>
		private BOL.Patient _patient;

		/// <summary>
		/// _dtSpecimen
		/// </summary>
		private DataTable _dtSpecimen;

		/// <summary>
		/// _drSpecimen
		/// </summary>
		private DataRow _drSpecimen;

		private bool _isSpecimenUidComplete = false;
		//private bool _isOverDefaultExpirationDays = false;
		private DateTime _previousExpirationDate;

		#endregion

		#region Constructors

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/26/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="3383">
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Valid Specimen object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3384">
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Specimen constructor
		/// </summary>
		public Specimen()
		{
			//Grab a "blank" datatable to build our new specimen with 
			this._dtSpecimen = DAL.Specimen.GetSpecimen(System.Guid.NewGuid());

			this._drSpecimen = this._dtSpecimen.NewRow();
			this.DivisionCode = Common.LogonUser.LogonUserDivisionCode;

			this.IsNew = true;

		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/26/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="2422">
		///		<ExpectedInput>data row</ExpectedInput>
		///		<ExpectedOutput>Specimen object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2732">
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates Specimen object and populates it with data from DataRow
		/// </summary>
		/// <param name="dr"></param>
		public Specimen(System.Data.DataRow dr)
		{
			this._dtSpecimen = dr.Table;
			this.LoadFromDataRow(dr);
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="2733">
		///		<ExpectedInput>SpecimenUid</ExpectedInput>
		///		<ExpectedOutput>specimen object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2734">
		///		<ExpectedInput>Invalid specimen Uid</ExpectedInput>
		///		<ExpectedOutput>New Object</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Specimen constructor with division code and specimen accession number parameters
		/// </summary>
		/// <param name="specimenUid"></param>
		public Specimen(string specimenUid)
		{
			this._dtSpecimen = DAL.Specimen.GetSpecimen(specimenUid);

			switch(_dtSpecimen.Rows.Count)
			{
				case 0:		
					this.IsNew = true;
					this.LoadFromDataRow(this._dtSpecimen.NewRow());
					this.SpecimenGuid = System.Guid.NewGuid();
					this.DivisionCode = Common.LogonUser.LogonUserDivisionCode;
					this.SpecimenUid = specimenUid;
					break;

				case 1:		//Populate with the data
					this.LoadFromDataRow(this._dtSpecimen.Rows[0]);
					this.PreviousExpirationDate = this.SpecimenExpirationDate;
					this.IsNew = false;
					this.IsDirty = false;
					break;

				default:
					string 	errorMessage = Common.StrRes.SysErrMsg.Common.TooManyDataRowsFoundException(string.Empty).ResString;
					throw new BOL.TooManyRecordsFoundException(errorMessage);
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="2781">
		///		<ExpectedInput>Valid Guid</ExpectedInput>
		///		<ExpectedOutput>Specimen Object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2782">
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>New object</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Instantiate new object from the patient speciment guid
		/// </summary>
		/// <param name="patientSpecimenGuid"></param>
		public Specimen(System.Guid patientSpecimenGuid)
		{
			this._dtSpecimen = DAL.Specimen.GetSpecimen(patientSpecimenGuid);

			switch(_dtSpecimen.Rows.Count)
			{
				case 0:		//No records found  -- create new?
					this.LoadFromDataRow(this._dtSpecimen.NewRow());
					this.SpecimenGuid = patientSpecimenGuid;

					this.IsNew = true;
					this.IsDirty = true;
					break;

				case 1:		//Populate with the data
					this.LoadFromDataRow(this._dtSpecimen.Rows[0]);
					this.PreviousExpirationDate = this.SpecimenExpirationDate;
					this.IsNew = false;
					this.IsDirty = false;
					break;

				default:
					string errorMessage = Common.StrRes.SysErrMsg.Common.TooManyDataRowsFoundException(string.Empty).ResString;
					throw new BOL.TooManyRecordsFoundException(errorMessage);
			}
		}

		#endregion

		#region Event Handlers

		//		/// <summary>
		//		/// ExtensionQuestions
		//		/// </summary>
		//		public event EventHandler ExtensionQuestions;
		//		
		//		/// <summary>
		//		/// OnExtensionQuestions
		//		/// </summary>
		//		/// <param name="questions"></param>
		//		protected virtual void OnExtensionQuestions(string questions)
		//		{
		//			if (ExtensionQuestions != null)
		//				ExtensionQuestions(questions, EventArgs.Empty);
		//		}
		//
		//		/// <summary>
		//		/// Is our Unacceptable Comment Valid?
		//		/// </summary>
		//		/// <param name="sender"></param>
		//		/// <param name="e"></param>
		//		private void UnacceptableCommentCheck(object sender, System.EventArgs e)
		//		{
		//			if (this.UnacceptableSpecimenIndicator)
		//			{
		//				if (this._unacceptableSpecimenComment.IsValid)
		//				{
		//					//RuleBroken(UNACCEPTABLE_SPECIMEN_INDICATOR, false);
		//				}
		//				else
		//				{
		//					//RuleBroken(UNACCEPTABLE_SPECIMEN_INDICATOR, true, "Comment is required");
		//				}
		//			}
		//		}
		//
		//		private void ExtensionQuestion(object sender, System.EventArgs e)
		//		{
		//			if (this._hasTransfusionHistory)
		//			{
		//
		//			}
		//		}
			
		#endregion

		#region Properties

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="2735">
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3386">
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Specimen Guid
		/// Implements BR_35.06
		/// </summary>
		public System.Guid SpecimenGuid
		{
			get
			{
				if (this._drSpecimen.IsNull(TABLE.PatientSpecimenGuid))
				{
					this._drSpecimen[TABLE.PatientSpecimenGuid] = System.Guid.NewGuid();
				}
				return (System.Guid)this._drSpecimen[TABLE.PatientSpecimenGuid];
			}
			set
			{
				this._drSpecimen[TABLE.PatientSpecimenGuid] = value;
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/2/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="3389">
		///		<ExpectedInput>bool</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3390">
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets or sets value indicating whether the specimen is being
		/// associated with a pending order in UC_16. This is required, as
		/// many BRs apply to entry from UC_16 only.
		/// </summary>
		public bool IsAssociatedWithPendingOrder
		{
			get
			{
				return this._isAssociatedWithPendingOrder;
			}
			set
			{
				this._isAssociatedWithPendingOrder = value;
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///	<TestCases>
		///
		///<Case type="0" testid ="2737">
		///		<ExpectedInput>Valid patient</ExpectedInput>
		///		<ExpectedOutput>Valid patient</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="9166"> 
		///		<ExpectedInput>Patient with TRs from conversion</ExpectedInput>
		///		<ExpectedOutput>HasAntibodyIdentification flag set to true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3396">
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Patient
		/// </summary>
		public BOL.Patient Patient
		{
			get
			{
				if (this._patient == null)
				{
					this._patient = new BOL.Patient(System.Guid.NewGuid());
				}
				return this._patient;
			}
			set
			{
				this._patient = value;
				this.IsDirty = true;
				//
				if (this._patient != null)
				{
					this._hasDiagnosticTests = DAL.Specimen.IsSpecimenAssociatedWithDiagnosticTest(this.SpecimenGuid);
					DataTable dtTRs = BOL.PatientTransfusionRequirement.GetTransfusionRequirementsForPatient(this.Patient.PatientGuid);
					DataRow[] drTRs = dtTRs.Select(Common.VbecsTables.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + " = '" + Common.Utility.GetTransfusionCategoryCodeFromEnum(Common.TransfusionRequirementCategoryCode.AntibodyIdentified) + "'");
					this._hasAntibodyIdentification = ((drTRs != null) && (drTRs.Length > 0));
					this._hasTransfusionHistory = BOL.PatientTransfusion.IsTransfusionHistoryforPatient(this.Patient.PatientGuid);
				}
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///	<TestCases>
		///
		///<Case type="0" testid ="3399">
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3400">
		///		<ExpectedInput>invalid specimen format</ExpectedInput>
		///		<ExpectedOutput>BrokenRule</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// SpecimenAccessionNumber
		/// </summary>
		public string SpecimenAccessionNumber
		{
			get
			{
				if (this._drSpecimen.IsNull(TABLE.SpecimenAccessionNumber))
				{
					this._drSpecimen[TABLE.SpecimenAccessionNumber] = String.Empty;
				}
				return this._drSpecimen[TABLE.SpecimenAccessionNumber].ToString();
			}
			set
			{
				this._drSpecimen[TABLE.SpecimenAccessionNumber] = value;
				this.IsDirty = true;
				
				//TODO: There are no broken rules implemented, so we're saving absolute CRAP to the DB!
				if ((value != null) && (Common.RegularExpressions.SpecimenAccessionNumber().IsMatch(value)==false))
				{
					RuleBroken(Method.GetCurrentMethod().Name, true);
				}
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///	<TestCases>
		///
		///<Case type="0" testid ="3418">
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3420">
		///		<ExpectedInput>invalid specimenuid format</ExpectedInput>
		///		<ExpectedOutput>BrokenRule</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// SpecimenUid, BR_35.06
		/// </summary>
		public string SpecimenUid
		{
			get
			{
				if (this._drSpecimen.IsNull(TABLE.SpecimenUid))
				{
					this._drSpecimen[TABLE.SpecimenUid] = String.Empty;
				}
				return this._drSpecimen[TABLE.SpecimenUid].ToString();
			}
			set
			{
				this._drSpecimen[TABLE.SpecimenUid] = value;
				this.IsDirty = true;

				if (value.Length == 0 || value.Length < 10 || value.Length  > 15)
				{
					RuleBroken(Method.GetCurrentMethod().Name, true);
					this._isSpecimenUidComplete = false;
				}
				else
				{
					RuleBroken(Method.GetCurrentMethod().Name, false);
					this._isSpecimenUidComplete = true;
				}				
			}
		}


		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="2736">
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>Valid string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3421">
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Specimen's VBECS Number
		/// </summary>
		public string VBECSSpecimenNumber
		{
			get
			{
				if (this._drSpecimen.IsNull(TABLE.VbecsSpecimenNumber))
				{
					this._drSpecimen[TABLE.VbecsSpecimenNumber] = String.Empty;
				}
				return this._drSpecimen[TABLE.VbecsSpecimenNumber].ToString();
			}
			set
			{
				this._drSpecimen[TABLE.VbecsSpecimenNumber] = value;
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="3422">
		///		<ExpectedInput>DateTime</ExpectedInput>
		///		<ExpectedOutput>DateTime</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3424">
		///		<ExpectedInput>future date</ExpectedInput>
		///		<ExpectedOutput>BrokenRule</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// SpecimenCollectionDate
		/// Implements BR_35.01
		/// </summary>
		public System.DateTime SpecimenCollectionDate
		{
			get
			{
				if (this._drSpecimen.IsNull(TABLE.SpecimenCollectionDate))
				{
					this._drSpecimen[TABLE.SpecimenCollectionDate] = VBECSDateTime.GetDivisionCurrentDateTime();
				}
				return (System.DateTime)this._drSpecimen[TABLE.SpecimenCollectionDate];
			}
			set
			{
				this._drSpecimen[TABLE.SpecimenCollectionDate] = value;
				this.IsDirty = true;
				//
				string collectionDateError = string.Empty;
				string receivedDateError = string.Empty;
				string expirationDateError = string.Empty;
				//
				this.CompareDates(ref collectionDateError, ref receivedDateError, ref expirationDateError);
				//
				RuleBroken("SpecimenCollectionDate",((SpecimenCollectionDate == DateTime.MinValue) || (collectionDateError.Length>0)));
				RuleBroken("SpecimenReceivedDate",((SpecimenReceivedDate == DateTime.MinValue) || (receivedDateError.Length>0)));
				RuleBroken("SpecimenExpirationDate",((SpecimenExpirationDate == DateTime.MinValue) || (expirationDateError.Length>0)));

				//BR_35.01
				this._isOverMaxDays = (value.Date.AddDays(this.MaximumSpecimenExpirationDays).AddHours(23).AddMinutes(59).AddSeconds(50) < this.SpecimenExpirationDate);
				RuleBroken("MaximumSpecimenExpirationDays", this._isOverMaxDays);

				
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="3432">
		///		<ExpectedInput>DateTime</ExpectedInput>
		///		<ExpectedOutput>DateTime</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3433">
		///		<ExpectedInput>Invalid DateTime</ExpectedInput>
		///		<ExpectedOutput>BrokenRule</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// SpecimenReceivedDate
		/// </summary>
		public System.DateTime SpecimenReceivedDate
		{
			get
			{
				if (this._drSpecimen.IsNull(TABLE.SpecimenReceivedDate))
				{
					this._drSpecimen[TABLE.SpecimenReceivedDate] = VBECSDateTime.GetDivisionCurrentDateTime();
				}
				return (System.DateTime)this._drSpecimen[TABLE.SpecimenReceivedDate];
			}
			set
			{
				this._drSpecimen[TABLE.SpecimenReceivedDate] = value;
				this.IsDirty = true;

				string collectionDateError = string.Empty;
				string receivedDateError = string.Empty;
				string expirationDateError = string.Empty;
				//
				this.CompareDates(ref collectionDateError, ref receivedDateError, ref expirationDateError);
				//
				RuleBroken("SpecimenCollectionDate",((SpecimenCollectionDate == DateTime.MinValue) || (collectionDateError.Length>0)));
				RuleBroken("SpecimenReceivedDate",((SpecimenReceivedDate == DateTime.MinValue) || (receivedDateError.Length>0)));
				RuleBroken("SpecimenExpirationDate",((SpecimenExpirationDate == DateTime.MinValue) || (expirationDateError.Length>0)));
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="3437">
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3441">
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// SpecimenCollectionTypeCode
		/// </summary>
		public string SpecimenSource
		{
			get
			{
				if (this._drSpecimen.IsNull(TABLE.SpecimenSource))
				{
					this._drSpecimen[TABLE.SpecimenSource] = String.Empty;
				}
				return this._drSpecimen[TABLE.SpecimenSource].ToString();
			}
			set
			{
				this._drSpecimen[TABLE.SpecimenSource] = value;
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///	<TestCases>
		///<Case type="0" testid ="2745">
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>Valid string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2746">
		///		<ExpectedInput>Empty string</ExpectedInput>
		///		<ExpectedOutput>broken rule</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Phlebotomist
		/// Implements BR_35.04
		/// </summary>
		public string Phlebotomist
		{
			get
			{
				if (this._drSpecimen.IsNull(TABLE.PhlebotomistId))
				{
					this._drSpecimen[TABLE.PhlebotomistId] = String.Empty;
				}
				return this._drSpecimen[TABLE.PhlebotomistId].ToString();
			}
			set
			{
				this._drSpecimen[TABLE.PhlebotomistId] = value;
				this.IsDirty = true;

				//BR_35.04
				//if (this._hasDiagnosticTests)
				//{
				//	RuleBroken(Method.GetCurrentMethod().Name, false);
				//}
				//else
				//{
				RuleBroken(Method.GetCurrentMethod().Name, ((value.Trim().Length < 3)||(value.Length > 35)));
				//}			
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///	<TestCases>
		///<Case type="0" testid ="3443">
		///		<ExpectedInput>bool</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3448">
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Indicates whether or not the specimen is maintainable.  Applies only to 
		/// Ordered Tests.
		/// </summary>
		public bool MaintainableSpecimenIndicator
		{
			get
			{
				if (this._drSpecimen.IsNull(TABLE.MaintainableSpecimenIndicator))
				{
					this._drSpecimen[TABLE.MaintainableSpecimenIndicator] = false;
				}
				return (bool) this._drSpecimen[TABLE.MaintainableSpecimenIndicator];
			}
			set
			{
				this._drSpecimen[TABLE.MaintainableSpecimenIndicator] = value;
				this.IsDirty = true;
			}
		}


		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/16/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="2027">
		///		<ExpectedInput>date/time</ExpectedInput>
		///		<ExpectedOutput>date/time</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="5996"> 
		///		<ExpectedInput>Valid collection date/time</ExpectedInput>
		///		<ExpectedOutput>Specimen expiration date 3 days in future</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1284">
		///		<ExpectedInput>Expiration date/time below collection date</ExpectedInput>
		///		<ExpectedOutput>BrokenRule</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Specimen's expiration date
		/// Implements BR_35.01, BR_35.02, BR_35.05, BR_35.11
		/// </summary>
		public System.DateTime SpecimenExpirationDate
		{
			get
			{
				//BR_35.02
				if (this._drSpecimen.IsNull(TABLE.SpecimenExpirationDate))
				{
					this._drSpecimen[TABLE.SpecimenExpirationDate] = this.SpecimenCollectionDate.Date.AddMinutes(-1).AddDays(Convert.ToDouble(Common.SpecimenExpirationDays.Minimum + 1));
				}
				return (System.DateTime)this._drSpecimen[TABLE.SpecimenExpirationDate];
			}

			set
			{

				this._drSpecimen[TABLE.SpecimenExpirationDate] = value;
				this.IsDirty = true;

				string collectionDateError = string.Empty;
				string receivedDateError = string.Empty;
				string expirationDateError = string.Empty;
				//
				this.CompareDates(ref collectionDateError, ref receivedDateError, ref expirationDateError);
				//
				RuleBroken("SpecimenCollectionDate",((SpecimenCollectionDate == DateTime.MinValue) || (collectionDateError.Length>0)));
				RuleBroken("SpecimenReceivedDate",((SpecimenReceivedDate == DateTime.MinValue) || (receivedDateError.Length>0)));
				RuleBroken("SpecimenExpirationDate",((SpecimenExpirationDate == DateTime.MinValue) || (expirationDateError.Length>0)));

			
				//BR_35.01
				this._isOverMaxDays = (this.SpecimenCollectionDate.Date.AddDays(this.MaximumSpecimenExpirationDays).AddHours(23).AddMinutes(59).AddSeconds(50) < value);
				RuleBroken("MaximumSpecimenExpirationDays", this._isOverMaxDays);


			
			}
		}

		

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/16/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="3450">
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3452">
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// SpecimenStatusCode
		/// </summary>
		public string SpecimenStatusCode
		{
			get
			{
				if (this._drSpecimen.IsNull(TABLE.SpecimenStatusCode))
				{
					this._drSpecimen[TABLE.SpecimenStatusCode] = "A";
				}
				return this._drSpecimen[TABLE.SpecimenStatusCode].ToString();
			}
			set
			{
				this._drSpecimen[TABLE.SpecimenStatusCode] = value;
				this.IsDirty = true;
			}
		}


		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///	<TestCases>
		///<Case type="0" testid ="2750">
		///		<ExpectedInput>bool</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2751">
		///		<ExpectedInput>True unacceptable specimen and AssociatedWithPendingOrder</ExpectedInput>
		///		<ExpectedOutput>broken rule</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Unacceptable specimen indicator
		/// Implements BR_35.17, BR_35.20
		/// </summary>
		public bool UnacceptableSpecimenIndicator
		{
			get
			{
				if (this._drSpecimen.IsNull(TABLE.UnacceptableSpecimenIndicator))
				{
					this._drSpecimen[TABLE.UnacceptableSpecimenIndicator] = false;
				}
				return (bool)this._drSpecimen[TABLE.UnacceptableSpecimenIndicator];
			}
			set
			{
				this._drSpecimen[TABLE.UnacceptableSpecimenIndicator] = value;
				this.IsDirty = true;
				//
				//BR_35.20 
				//New specimen cannot be unacceptable
				RuleBroken(Method.GetCurrentMethod().Name, (value && this._isAssociatedWithPendingOrder));
				//BR_35.17
				RuleBroken("CannedCommentGuid",(value && ((this.UnacceptableSpecimenComment == null)||(this.UnacceptableSpecimenComment.CannedCommentGuid == Guid.Empty)||(this.UnacceptableSpecimenComment.Comments == null))));				
				RuleBroken("Details",((this.UnacceptableSpecimenIndicator) && ((this.UnacceptableSpecimenComment != null)&&(this.UnacceptableSpecimenComment.Comments != null)&&(this.UnacceptableSpecimenComment.Comments.ToUpper()=="OTHER")&&(this.UnacceptableSpecimenDetails.Trim().Length==0))));
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///	<TestCases>
		///<Case type="0" testid ="3454">
		///		<ExpectedInput>bool</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3455">
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// PostTransfusionIndicator
		/// </summary>
		public bool PostTransfusionIndicator
		{
			get
			{
				if (this._drSpecimen.IsNull(TABLE.PostTransfusionIndicator))
				{
					this._drSpecimen[TABLE.PostTransfusionIndicator] = false;
				}
				return (bool)this._drSpecimen[TABLE.PostTransfusionIndicator];
			}
			set
			{
				this._drSpecimen[TABLE.PostTransfusionIndicator] = value;
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///	<TestCases>
		///<Case type="0" testid ="3464">
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3481">
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// DivisionCode
		/// </summary>
		public string DivisionCode
		{
			get
			{
				if (this._drSpecimen.IsNull(TABLE.DivisionCode))
				{
					this._drSpecimen[TABLE.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
				}
				return this._drSpecimen[TABLE.DivisionCode].ToString();
			}
			set
			{
				this._drSpecimen[TABLE.DivisionCode] = value;
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///	<TestCases>
		///<Case type="0" testid ="3482">
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3483">
		///		<ExpectedInput>Empty String</ExpectedInput>
		///		<ExpectedOutput>broken rule</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Processing Tech ID
		/// </summary>
		public string ProcessingTechId
		{
			get
			{
				if (this._drSpecimen.IsNull(TABLE.EntryTechId))
				{
					this._drSpecimen[TABLE.EntryTechId] = string.Empty;
				}
				return this._drSpecimen[TABLE.EntryTechId].ToString();
			}
			set
			{
				this._drSpecimen[TABLE.EntryTechId] = value;
				RuleBroken(Method.GetCurrentMethod().Name, (value.Trim().Length == 0));
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///	<TestCases>
		///<Case type="0" testid ="2752">
		///		<ExpectedInput>Unacceptable specimen comment</ExpectedInput>
		///		<ExpectedOutput>Valid unacceptable specimen comment</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="3497">
		///		<ExpectedInput>Invalid Comment</ExpectedInput>
		///		<ExpectedOutput>Broken Rule</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Unacceptable specimen comment
		/// </summary>
		public BOL.CannedComment UnacceptableSpecimenComment
		{
			get
			{
				if (this._unacceptableSpecimenComment == null)
				{
					this._unacceptableSpecimenComment = new BOL.CannedComment(System.Guid.NewGuid());
				}
				return this._unacceptableSpecimenComment;
			}
			set
			{
				this.IsDirty = true;
				this._unacceptableSpecimenComment = value;
				//
				//BR_35.17
				RuleBroken("CannedCommentGuid",((this.UnacceptableSpecimenIndicator) && ((this.UnacceptableSpecimenComment == null)||(this.UnacceptableSpecimenComment.CannedCommentGuid == Guid.Empty)||(this.UnacceptableSpecimenComment.Comments == null))));
				RuleBroken("Details",((this.UnacceptableSpecimenIndicator) && ((this.UnacceptableSpecimenComment != null)&&(this.UnacceptableSpecimenComment.Comments != null)&&(this.UnacceptableSpecimenComment.Comments.ToUpper()=="OTHER")&&(this.UnacceptableSpecimenDetails.Trim().Length==0))));
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///	<TestCases>
		///<Case type="0" testid ="3500">
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="3501">
		///		<ExpectedInput>empty string</ExpectedInput>
		///		<ExpectedOutput>Broken Rule</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// UnacceptableSpecimenDetails
		/// </summary>
		public string UnacceptableSpecimenDetails
		{
			get
			{
				if (this._drSpecimen.IsNull(TABLE.SpecimenComments))
				{
					this._drSpecimen[TABLE.SpecimenComments] = string.Empty;
				}
				return this._drSpecimen[TABLE.SpecimenComments].ToString();
			}
			set
			{
				this.IsDirty = true;
				this._drSpecimen[TABLE.SpecimenComments] = value;
				//
				//BR_35.17
				RuleBroken("CannedCommentGuid",((this.UnacceptableSpecimenIndicator) && ((this.UnacceptableSpecimenComment == null)||(this.UnacceptableSpecimenComment.CannedCommentGuid == Guid.Empty)||(this.UnacceptableSpecimenComment.Comments == null))));
				RuleBroken("Details",((this.UnacceptableSpecimenIndicator) && ((this.UnacceptableSpecimenComment != null)&&(this.UnacceptableSpecimenComment.Comments != null)&&(this.UnacceptableSpecimenComment.Comments.ToUpper()=="OTHER")&&(this.UnacceptableSpecimenDetails.Trim().Length==0))));
			}
		}

		//		///<Developers>
		//		///	<Developer>Luke Meyer</Developer>
		//		///</Developers>
		//		///<SiteName>Hines OIFO</SiteName>
		//		///<CreationDate>5/2/2003</CreationDate>
		//		///	<TestCases>
		//		///<Case type="0" testid ="2753">
		//		///		<ExpectedInput>Valid extended specimen comment</ExpectedInput>
		//		///		<ExpectedOutput>Valid extended specimen comment</ExpectedOutput>
		//		///	</Case>
		//		///
		//		///</TestCases>
		//		///<Update></Update>
		//		///<ArchivePlan></ArchivePlan>
		//		///<Interfaces></Interfaces>
		//		/// <summary>
		//		/// Extended specimen comment
		//		/// </summary>
		//		public BOL.CannedComment ExtendedSpecimenComment
		//		{
		//			get
		//			{
		//				return this._extendedSpecimenComment;
		//			}
		//			set
		//			{
		//				this._extendedSpecimenComment = value;
		//				this.IsDirty = true;
		//			}
		//		}
		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///
		///	<TestCases>
		///<Case type="0" testid ="2754">
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Valid boolean</ExpectedOutput>ut>
		///	</Case>
		///<Case type="1" testid ="3508">
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>ut>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Expired specimen indicator
		/// </summary>
		public bool IsExpired
		{
			get
			{
				return this.SpecimenExpirationDate.Date < VBECSDateTime.GetDivisionCurrentDateTime().Date;
			}
		}
		
		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///
		///	<TestCases>
		///<Case type="0" testid ="3509">
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Valid boolean</ExpectedOutput>ut>
		///	</Case>
		///<Case type="1" testid ="3536">
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>ut>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// IsSpecimenUidComplete
		/// </summary>
		public bool IsSpecimenUidComplete
		{
			get
			{
				return this._isSpecimenUidComplete;
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///	<TestCases>
		///<Case type="0" testid ="2755">
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Valid boolean</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="3537">
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Extension indicator
		/// Implements BR_35.05
		/// </summary>
		public bool IsExtended
		{
			get
			{
				return (this.SpecimenExpirationDate.Date > this.SpecimenCollectionDate.AddDays(Convert.ToDouble(Common.SpecimenExpirationDays.Minimum)).Date);
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///	<TestCases>
		///<Case type="0" testid ="2756">
		///		<ExpectedInput>Valid short</ExpectedInput>
		///		<ExpectedOutput>Valid short</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="3538">
		///		<ExpectedInput>inValid short</ExpectedInput>
		///		<ExpectedOutput>broken rule</ExpectedOutput>
		///	</Case>
		///	</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Maximum Specimen Expiration Days
		/// </summary>
		public short MaximumSpecimenExpirationDays
		{
			get
			{
				return Convert.ToInt16(this._maximumSpecimenExpirationDays);
			}

			set
			{
				RuleBroken(Method.GetCurrentMethod().Name, ((value < Convert.ToInt16(Common.SpecimenExpirationDays.Minimum)) || (value > Convert.ToInt16(Common.SpecimenExpirationDays.Maximum))));
				_maximumSpecimenExpirationDays = value;
			}
		}


		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/22/2003</CreationDate>
		///	<TestCases>
		///	
		///<Case type="0" testid ="3499"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Valid data table</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="3539"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// TestResults
		/// </summary>
		public DataTable TestResults
		{
			get
			{
//				if (this._testResults == null)
//				{
//					this._testResults = this.GetTestResultsForSpecimen();
//				}
				return this.GetTestResultsForSpecimen();
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/22/2003</CreationDate>
		///	<TestCases>
		///	
		///<Case type="0" testid ="3496"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Valid boolean</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="3540"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// HasAntibodyIdentification
		/// </summary>
		public bool HasAntibodyIdentification
		{
			get
			{
				return this._hasAntibodyIdentification;
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/22/2003</CreationDate>
		///	<TestCases>
		///	
		///<Case type="0" testid ="3498"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Valid boolean</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="3541"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// HasTransfusionHistory
		/// </summary>
		public bool HasTransfusionHistory
		{
			get
			{
				return this._hasTransfusionHistory;
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/22/2003</CreationDate>
		///	<TestCases>
		///	
		///<Case type="0" testid ="3542"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Valid boolean</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="3545"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Indicates whether the speciman has current units assigned to it
		/// Implements BR_94.03
		/// </summary>
		public bool HasAssignedUnits
		{
			get
			{
				return (BOL.OrderedUnit.GetOrderedUnitsForSpecimen(this.SpecimenGuid).Count>0);
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/7/2003</CreationDate>
		///	<TestCases>
		///	<Case type="0" testid ="1473"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Boolean value</ExpectedOutput>
		///	</Case>
		///	<Case type="1" testid ="3546"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// PreviouslySavedAsUnacceptable
		/// Implemements BR_35.10
		/// </summary>
		public bool PreviouslySavedAsUnacceptable
		{
			get
			{
				return this._previouslySavedAsUnacceptable;
			}
		}
		
		
		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/5/2003</CreationDate>
		///	<TestCases>
		///	
		///<Case type="0" testid ="1542"> 
		///		<ExpectedInput>Valid Date/Time</ExpectedInput>
		///		<ExpectedOutput>Valid Date/Time</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="3547"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// PreviousExpirationDate
		/// Implements BR_35.10, BR_35.11
		/// </summary>
		public DateTime PreviousExpirationDate
		{
			//BR_35.11
			get
			{
				return this._previousExpirationDate;
			}

			set
			{
				_previousExpirationDate = value;
			}
		}
		
		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/5/2003</CreationDate>
		///	<TestCases>
		///	
		///<Case type="0" testid ="3548"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="3549"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// ABORhTested
		/// Implements BR_56.17
		/// </summary>
		public bool ABORhTested
		{
			get
			{
                bool aboTested = false;
                bool rhTested = false;

                // BR_56.17
                // CR3283
                DataRow[] drABOs = this.TestResults.Select(Common.VbecsTables.BloodTestType.BloodTestTypeId + " = " + (int)Common.TestType.ABOInterp + " OR BloodTestTypeId = " + (int)Common.TestType.ABOInterpTAS);
                DataRow[] drRhs = this.TestResults.Select(Common.VbecsTables.BloodTestType.BloodTestTypeId + " = " + (int)Common.TestType.RhInterp + " OR BloodTestTypeId = " + (int)Common.TestType.RhInterpTAS);

                for (int i = 0; i < drABOs.Length; i++)
                {
                    if (drABOs[i][Common.VbecsTables.BloodUnitTest.TestResultId] != System.DBNull.Value)
                    {
                        aboTested = true;
                        break;
                    }
                }
                for (int i = 0; i < drRhs.Length; i++)
                {
                    if (drRhs[i][Common.VbecsTables.BloodUnitTest.TestResultId] != System.DBNull.Value)
                    {
                        rhTested = true;
                        break;
                    }
                }
                return (aboTested && rhTested);

			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/5/2003</CreationDate>
		///	<TestCases>
		///	
		///<Case type="0" testid ="3550"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="3551"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// TASTested
		/// </summary>
		public bool TASTested
		{
			get
			{
				DataTable dtTest = BOL.OrderedTest.GetOrderedTests(this.SpecimenGuid);
				DataRow [] dtRowArray = dtTest.Select(Common.VbecsTables.OrderedTest.OrderableTestId  + " = " + (int) Common.OrderableTest.TAS);
				foreach (DataRow dtResultRow in dtRowArray)
				{
					Common.OrderStatus orderStatus = Common.Utility.GetOrderStatusCodeFromString( dtResultRow[Common.VbecsTables.OrderedTest.OrderStatusCode].ToString() );
					if (orderStatus == Common.OrderStatus.Complete)
					{
						//CR 2822
						DataTable dtTestResults = this.GetTestResultsForSpecimen();
						if (   dtTestResults.Select(Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " + Convert.ToInt32(Common.TestType.ABOInterpTAS)).Length > 0
							&& dtTestResults.Select(Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " + Convert.ToInt32(Common.TestType.RhInterpTAS)).Length > 0
							&& dtTestResults.Select(Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " + Convert.ToInt32(Common.TestType.AntibodyScreenTAS)).Length > 0)
						{
							return true;
						}
					}
				}
				return false;
			}
		}

		
		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/5/2003</CreationDate>
		///	<TestCases>
		///	
		///<Case type="0" testid ="3552"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Common.TypingTestResults</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="3553"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// RepeatABORhTestsMatch BR_3.40
		/// </summary>
		public Common.TypingTestResults RepeatABORhTestsMatch
		{
			get
			{
				DataTable dt = this.TestResults;
				DataRow [] drABOs	= dt.Select(Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " + (int) Common.TestType.ABOInterp + " OR " + Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " + (int) Common.TestType.ABOInterpTAS + " OR " + Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " + (int) Common.TestType.ABORepeat, Common.VbecsTables.SpecimenTest.TestDate + " DESC");
				DataRow [] drRhs	= dt.Select(Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " + (int) Common.TestType.RhInterp + " OR " + Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " + (int) Common.TestType.RhInterpTAS + " OR " + Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " + (int) Common.TestType.RhRepeat, Common.VbecsTables.SpecimenTest.TestDate + " DESC");
				//
				// Now, get the TestResultId and make sure that it matches the historical ABO/Rh.  If multiple tests have 
				// been performed on the specimen, ALL must be compared:
				//
				// compare each test result on the current specimen to test results from all previous
				//
				// first, ABO tests
				if(drABOs.Length == 0)
				{
					// Couldn't have gotten here if historic results already proven to be true, so go on!
					return Common.TypingTestResults.NoResults;
				}
				else
				{
					string firstResult = string.Empty;
					for (int i=0; i<drABOs.Length; i++)
					{
						// check for justification
						if ( ( (bool) drABOs[i][Common.VbecsTables.SpecimenTest.AboRhChangeIndicator]) == true)
						{
							return Common.TypingTestResults.Match;
						}

						if (i==0 || firstResult == string.Empty)
						{
							if (drABOs[i][Common.VbecsTables.BloodUnitTest.TestResultId] != System.DBNull.Value)
							{
								firstResult = ((string) drABOs[i][Common.VbecsTables.BloodUnitTest.TestResultId]).Trim();
							}
							
						}
						else
						{
							if (drABOs[i][Common.VbecsTables.BloodUnitTest.TestResultId] != System.DBNull.Value &&
								!firstResult.Equals( ((string) drABOs[i][Common.VbecsTables.BloodUnitTest.TestResultId]).Trim() ))
							{
								return Common.TypingTestResults.Discrepancy;
							}
						}
					}
				}

				// now, Rh tests
				if(drRhs.Length == 0)
				{
					return Common.TypingTestResults.NoResults;
				}
				else
				{
					string firstResult = string.Empty;
					for (int i=0; i<drRhs.Length; i++)
					{
						
						if (i==0 || firstResult == string.Empty)
						{
							if (drRhs[i][Common.VbecsTables.BloodUnitTest.TestResultId] != System.DBNull.Value)
							{
								firstResult = ((string) drRhs[i][Common.VbecsTables.BloodUnitTest.TestResultId]).Trim();
							}
						}
						else
						{
							if (drRhs[i][Common.VbecsTables.BloodUnitTest.TestResultId] != System.DBNull.Value &&
								!firstResult.Equals( ((string) drRhs[i][Common.VbecsTables.BloodUnitTest.TestResultId]).Trim() ))
							{
								return Common.TypingTestResults.Discrepancy;
							}
						}
					}
				}
				return Common.TypingTestResults.Match;
			}
		}
		

		#endregion

		#region Methods



		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="2761">
		///		<ExpectedInput>Valid specimen with no exceptions</ExpectedInput>
		///		<ExpectedOutput>Success indicator = 0</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2762">
		///		<ExpectedInput>ExceptionReports, UpdateFunction</ExpectedInput>
		///		<ExpectedOutput>RowVersionException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="7549"> 
		///		<ExpectedInput>Valid specimen with extended expiration exception</ExpectedInput>
		///		<ExpectedOutput>Success indicator = 0</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="0" testid ="7550"> 
		///		<ExpectedInput>Valid specimen with reactivated specimen exception</ExpectedInput>
		///		<ExpectedOutput>Success indicator = 0</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Adds/updates a specimen and/or specimen comment
		/// Implements BR_35.15 via database triggers
		/// </summary>
		/// <param name="specimenReactivatedException">exception</param>
		/// <param name="specimenExtendedException">exception</param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <returns></returns>
		public int UpdateSpecimen(BOL.ExceptionReport specimenReactivatedException, 
			BOL.ExceptionReport specimenExtendedException, Common.UpdateFunction lastUpdateFunctionId)
		{
			DataSet dsSpecimen = this.GetSpecimenSaveDataset();

			DataTable dtExceptionSpecimen = null;
			System.Data.DataTable dtWorkloadEvents = null;

			if (specimenReactivatedException != null)
			{
				dtExceptionSpecimen = specimenReactivatedException.ExceptionData.Table.Clone();
				dtExceptionSpecimen.Rows.Add(specimenReactivatedException.ExceptionData.ItemArray);
			}

			if (specimenExtendedException != null)
			{
				if (dtExceptionSpecimen == null)
				{
					dtExceptionSpecimen = specimenExtendedException.ExceptionData.Table.Clone();
				}
				dtExceptionSpecimen.Rows.Add(specimenExtendedException.ExceptionData.ItemArray);
			}
			System.Guid [] specimenGuids = new Guid[1];
			specimenGuids[0] = this.SpecimenGuid;
			//CR 2171 we dont want to collect workload if we are just invalidating.
			if(lastUpdateFunctionId != Common.UpdateFunction.UC094FrmInvalidatePatientTestResults)
			{
				dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.MaintainSpecimen, specimenGuids, Common.WorkloadTransactionType.Miscellanious, false);
			}
			//BR_35.15
			int returnValue = DAL.Specimen.UpdateSpecimen(dsSpecimen.Tables[Specimen.SPECIMENTABLE], dtExceptionSpecimen, dtWorkloadEvents, lastUpdateFunctionId);
			return returnValue;
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="3554">
		///		<ExpectedInput>ExceptionReport, UpdateFunction</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3555">
		///		<ExpectedInput>ExceptionReport, UpdateFunction</ExpectedInput>
		///		<ExpectedOutput>RowVersionException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// InsertSpecimen
		/// </summary>
		/// <param name="specimenExtendedException"></param>
		/// <param name="lastUpdateFunctionId"></param>
		/// <returns></returns>
		public int InsertSpecimen(BOL.ExceptionReport specimenExtendedException, Common.UpdateFunction lastUpdateFunctionId)
		{
			DataSet dsSpecimen = this.GetSpecimenSaveDataset();

			DataTable dtExceptionSpecimen = null;

			if (specimenExtendedException != null)
			{
				dtExceptionSpecimen = specimenExtendedException.ExceptionData.Table.Clone();
				dtExceptionSpecimen.Rows.Add(specimenExtendedException.ExceptionData.ItemArray);
			}
			
			System.Guid [] specimenGuids = new Guid[1];
			specimenGuids[0] = this.SpecimenGuid;

			DataTable dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.MaintainSpecimen, specimenGuids, Common.WorkloadTransactionType.Miscellanious, false);

			int returnValue = DAL.Specimen.InsertSpecimen(dsSpecimen.Tables[Specimen.SPECIMENTABLE], dtExceptionSpecimen, dtWorkloadEvents, lastUpdateFunctionId);
			return returnValue;
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/8/2003</CreationDate>
		///<TestCases>
		///<Case type="0" testid ="3480"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Dataset wih two tables</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="3556"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetSpecimenSaveDataset
		/// </summary>
		/// <returns></returns>
		public DataSet GetSpecimenSaveDataset()
		{
			DataSet dsSpecimen = new DataSet();
			DataTable dtSpecimen = DAL.Specimen.GetEmptyPatientSpecimenTable(this.IsNew);
			DataRow drSpecimen = this.LoadDataRowFromThis(dtSpecimen.NewRow());
			dtSpecimen.Rows.Add(drSpecimen);

			//			DataTable dtSpecimenComment = DAL.SpecimenComment.GetEmptyPatientSpecimenCommentTable(false);
			//			DataTable dtInsertSpecimenComment = DAL.SpecimenComment.GetEmptyPatientSpecimenCommentTable(true);

			//			if (this.IsExtended)
			//			{
			//				if (this.ExtendedSpecimenComment.IsDirty && this.ExtendedSpecimenComment.IsValid)
			//				{
			//					DataRow drComment = this.ExtendedSpecimenComment.LoadDataRowFromThis(dtSpecimenComment.NewRow());
			//					drComment[TABLE.PatientSpecimenGuid] = this.SpecimenGuid;
			//					dtSpecimenComment.Rows.Add(drComment);
			//				}
			//			}
			//			if (this.UnacceptableSpecimenIndicator)
			//			{
			//				if (this.UnacceptableSpecimenComment.IsDirty && this.UnacceptableSpecimenComment.IsValid)
			//				{
			//					if (!this.UnacceptableSpecimenComment.IsNew)
			//					{
			//						DataRow drComment = this.UnacceptableSpecimenComment.LoadDataRowFromThis(dtSpecimenComment.NewRow());
			//						drComment[TABLE.PatientSpecimenGuid] = this.SpecimenGuid;
			//						dtSpecimenComment.Rows.Add(drComment);
			//					}
			//					else
			//					{
			//						DataRow drComment = this.UnacceptableSpecimenComment.LoadDataRowFromThis(dtInsertSpecimenComment.NewRow());
			//						drComment[TABLE.PatientSpecimenGuid] = this.SpecimenGuid;
			//						dtInsertSpecimenComment.Rows.Add(drComment);
			//					}
			//				}
			//			}
			dtSpecimen.TableName = Specimen.SPECIMENTABLE;
			//			dtSpecimenComment.TableName = Specimen.SPECIMENCOMMENTTABLE;
			//			dtInsertSpecimenComment.TableName = Specimen.INSERTSPECIMENCOMMENTTABLE;

			dsSpecimen.Tables.Add(dtSpecimen);
			//			dsSpecimen.Tables.Add(dtSpecimenComment);
			//			dsSpecimen.Tables.Add(dtInsertSpecimenComment);

			return dsSpecimen;
		}

		/// <summary>
		/// LoadFromDataRow
		/// </summary>
		/// <param name="dtRow"></param>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{

			this._drSpecimen = DAL.Specimen.GetEmptyPatientSpecimenTable(!dtRow.Table.Columns.Contains("RowVersion")).NewRow();


			if (dtRow.Table.Columns.Contains(TABLE.DivisionCode))
			{
				if (!dtRow.IsNull(TABLE.DivisionCode))
				{
					this._drSpecimen[TABLE.DivisionCode] = dtRow[TABLE.DivisionCode];
				}
			}

			if (dtRow.Table.Columns.Contains(TABLE.SpecimenExpirationDate))
			{
				if (!dtRow.IsNull(TABLE.SpecimenExpirationDate))
				{
					this._drSpecimen[TABLE.SpecimenExpirationDate] = dtRow[TABLE.SpecimenExpirationDate];
				}
			}

			//			if (dtRow.Table.Columns.Contains(TABLE.ExtendedExpirationDate))
			//			{
			//				if (!dtRow.IsNull(TABLE.ExtendedExpirationDate))
			//				{
			//					this._drSpecimen[TABLE.ExtendedExpirationDate] = dtRow[TABLE.ExtendedExpirationDate];
			//				}
			//			}

//			if (dtRow.Table.Columns.Contains(TABLE.HemolysisInspectionCode))
//			{
//				if (!dtRow.IsNull(TABLE.HemolysisInspectionCode))
//				{
//					this._drSpecimen[TABLE.HemolysisInspectionCode] = dtRow[TABLE.HemolysisInspectionCode];
//				}
//			}

			if (dtRow.Table.Columns.Contains(TABLE.PatientGuid))
			{
				if (!dtRow.IsNull(TABLE.PatientGuid))
				{
					this._drSpecimen[TABLE.PatientGuid] = dtRow[TABLE.PatientGuid];
					this._patient = new BOL.Patient((System.Guid)this._drSpecimen[TABLE.PatientGuid]);
				}
			}

			if (dtRow.Table.Columns.Contains(TABLE.PatientSpecimenGuid))
			{
				if (!dtRow.IsNull(TABLE.PatientSpecimenGuid))
				{
					this._drSpecimen[TABLE.PatientSpecimenGuid] = dtRow[TABLE.PatientSpecimenGuid];
				}
			}

			if (dtRow.Table.Columns.Contains(TABLE.PhlebotomistId))
			{
				if (!dtRow.IsNull(TABLE.PhlebotomistId))
				{
					this._drSpecimen[TABLE.PhlebotomistId] = dtRow[TABLE.PhlebotomistId];
				}
			}

			if (dtRow.Table.Columns.Contains(TABLE.PostTransfusionIndicator))
			{
				if (!dtRow.IsNull(TABLE.PostTransfusionIndicator))
				{
					this._drSpecimen[TABLE.PostTransfusionIndicator] = dtRow[TABLE.PostTransfusionIndicator];
				}
			}

			if (dtRow.Table.Columns.Contains(TABLE.SpecimenAccessionNumber))
			{
				if (!dtRow.IsNull(TABLE.SpecimenAccessionNumber))
				{
					this._drSpecimen[TABLE.SpecimenAccessionNumber] = dtRow[TABLE.SpecimenAccessionNumber];
				}
			}

			if (dtRow.Table.Columns.Contains(TABLE.SpecimenCollectionDate))
			{
				if (!dtRow.IsNull(TABLE.SpecimenCollectionDate))
				{
					this._drSpecimen[TABLE.SpecimenCollectionDate] = dtRow[TABLE.SpecimenCollectionDate];
				}
			}

			if (dtRow.Table.Columns.Contains(TABLE.SpecimenReceivedDate))
			{
				if (!dtRow.IsNull(TABLE.SpecimenReceivedDate))
				{
					this._drSpecimen[TABLE.SpecimenReceivedDate] = dtRow[TABLE.SpecimenReceivedDate];
				}
			}

			if (dtRow.Table.Columns.Contains(TABLE.SpecimenSource))
			{
				if (!dtRow.IsNull(TABLE.SpecimenSource))
				{
					this._drSpecimen[TABLE.SpecimenSource] = dtRow[TABLE.SpecimenSource];
				}
			}

			if (dtRow.Table.Columns.Contains(TABLE.SpecimenStatusCode))
			{
				if (!dtRow.IsNull(TABLE.SpecimenStatusCode))
				{
					this._drSpecimen[TABLE.SpecimenStatusCode] = dtRow[TABLE.SpecimenStatusCode];
				}
			}

			if (dtRow.Table.Columns.Contains(TABLE.SpecimenUid))
			{
				if (!dtRow.IsNull(TABLE.SpecimenUid))
				{
					this._drSpecimen[TABLE.SpecimenUid] = dtRow[TABLE.SpecimenUid];
				}
			}

			if (dtRow.Table.Columns.Contains(TABLE.UnacceptableSpecimenIndicator))
			{
				if (!dtRow.IsNull(TABLE.UnacceptableSpecimenIndicator))
				{
					this._drSpecimen[TABLE.UnacceptableSpecimenIndicator] = dtRow[TABLE.UnacceptableSpecimenIndicator];
					_previouslySavedAsUnacceptable = (bool) dtRow[TABLE.UnacceptableSpecimenIndicator];
				}
			}

			if (dtRow.Table.Columns.Contains(TABLE.VbecsSpecimenNumber))
			{
				if (!dtRow.IsNull(TABLE.VbecsSpecimenNumber))
				{
					this._drSpecimen[TABLE.VbecsSpecimenNumber] = dtRow[TABLE.VbecsSpecimenNumber];
				}
			}

			if (dtRow.Table.Columns.Contains(TABLE.MaintainableSpecimenIndicator))
			{
				if (!dtRow.IsNull(TABLE.MaintainableSpecimenIndicator))
				{
					this.MaintainableSpecimenIndicator = (bool) dtRow[TABLE.MaintainableSpecimenIndicator];
				}
			}

			if (dtRow.Table.Columns.Contains(TABLE.RowVersion))
			{
				if (!dtRow.IsNull(TABLE.RowVersion))
				{
					this.RowVersion = (byte[]) dtRow[TABLE.RowVersion];
				}
			}

			//this.LoadSpecimenComments();
			if (dtRow.Table.Columns.Contains(TABLE.CannedCommentGuid))
			{
				if (!dtRow.IsNull(TABLE.CannedCommentGuid))
				{
					this._unacceptableSpecimenComment = new BOL.CannedComment((Guid)dtRow[TABLE.CannedCommentGuid]);
				}
			}

			if (dtRow.Table.Columns.Contains(TABLE.SpecimenComments))
			{
				if (!dtRow.IsNull(TABLE.SpecimenComments))
				{
					this.UnacceptableSpecimenDetails = dtRow[TABLE.SpecimenComments].ToString();
				}
			}
			
			BOL.Division div = new BOL.Division(this.DivisionCode);
			this.MaximumSpecimenExpirationDays = div.MaximumSpecimenExpirationDays;

			if (this.Patient == null)
			{
				this._hasDiagnosticTests = false;
				this._hasAntibodyIdentification = false;
				this._hasTransfusionHistory = false;
			}

			else
			{
				this._hasDiagnosticTests = DAL.Specimen.IsSpecimenAssociatedWithDiagnosticTest(this.SpecimenGuid);
				DataTable dtTRs = BOL.PatientTransfusionRequirement.GetTransfusionRequirementsForPatient(this.Patient.PatientGuid);
				DataRow[] drTRs = dtTRs.Select(Common.VbecsTables.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + " = '" + Common.Utility.GetTransfusionCategoryCodeFromEnum(Common.TransfusionRequirementCategoryCode.AntibodyIdentified) + "'");
				this._hasAntibodyIdentification = ((drTRs != null) && (drTRs.Length > 0));
				this._hasTransfusionHistory = BOL.PatientTransfusion.IsTransfusionHistoryforPatient(this.Patient.PatientGuid);
			}		
			
			//			/*
			//			SpecimenSource comes from VISTA
			//			SpecimenStatusCode	comes from UC 16
			//			*/

			//			//BR 35.11			
			//			if (dtRow[TABLE.ExtendedExpirationDate] != System.DBNull.Value)  
			//				//Put in extended expiration date, if found, first
			//			{
			//				this.SpecimenExpirationDate = (System.DateTime)dtRow[TABLE.ExtendedExpirationDate];
			//			}			
			//			
			if (dtRow[TABLE.SpecimenExpirationDate] != System.DBNull.Value) 
			{
				this.SpecimenExpirationDate = (System.DateTime)dtRow[TABLE.SpecimenExpirationDate];
			}
			else	
				//If specimen expiration is null, put in default expiration date
			{
				//Actually... don't cos it breaks UC_35. If Expiration Date is not set, neither is Collection Date, so default is wrong!
				//this.SpecimenExpirationDate = this.SpecimenCollectionDate.Date.AddMinutes(-1).AddDays(Convert.ToDouble(Common.SpecimenExpirationDays.Minimum + 1));
			}

			if (dtRow.Table.Columns.Contains(TABLE.EntryTechId))
			{
				if (!dtRow.IsNull(TABLE.EntryTechId))
				{
					this._drSpecimen[TABLE.EntryTechId] = dtRow[TABLE.EntryTechId];
				}
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/5/2003</CreationDate>
		///	<TestCases>
		///	
		///<Case type="0" testid ="3332"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Data table of test results</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="3557"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Data table of test results</ExpectedOutput>
		///	</Case>
		///	</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Returns a DataTable of TestResults from a specimen GUID
		/// </summary>
		/// <returns></returns>
		public DataTable GetTestResultsForSpecimen()
		{
			return DAL.Specimen.GetTestResultsForSpecimen(this.SpecimenGuid);
		}


		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/5/2003</CreationDate>
		///	<TestCases>
		///	
		///<Case type="0" testid ="3558"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>DataRow</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="3559"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Loads a data row from an object
		/// </summary>
		/// <param name="dtRow"></param>
		/// <returns></returns>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			if(dtRow.Table.Columns.Contains(TABLE.PatientSpecimenGuid))
			{
				dtRow[TABLE.PatientSpecimenGuid] = this.SpecimenGuid;
			}

			if(dtRow.Table.Columns.Contains(TABLE.PatientGuid))
			{
				dtRow[TABLE.PatientGuid] = this.Patient.PatientGuid;
			}

			if(dtRow.Table.Columns.Contains(TABLE.SpecimenAccessionNumber))
			{
				dtRow[TABLE.SpecimenAccessionNumber] =  this.SpecimenAccessionNumber;
			}

			if(dtRow.Table.Columns.Contains(TABLE.SpecimenUid))
			{
				dtRow[TABLE.SpecimenUid] =  this.SpecimenUid;
			}

			//				//BR 35.11
			//				if (this.IsNew)
			//				{
			//					dtRow[TABLE.VbecsSpecimenNumber] =  this.SpecimenAccessionNumber;
			//				}
			//
			//				if (this.VBECSSpecimenNumber == string.Empty)
			//				{
			//					dtRow[TABLE.VbecsSpecimenNumber] =  DBNull.Value;
			//				}
			//				else
			//				{
			//					dtRow[TABLE.VbecsSpecimenNumber] =  this.VBECSSpecimenNumber;
			//				}
			//
			if(dtRow.Table.Columns.Contains(TABLE.SpecimenCollectionDate))
			{
				dtRow[TABLE.SpecimenCollectionDate] =  this.SpecimenCollectionDate;
			}

			if(dtRow.Table.Columns.Contains(TABLE.SpecimenReceivedDate))
			{
				dtRow[TABLE.SpecimenReceivedDate] =  this.SpecimenReceivedDate;
			}

			if(dtRow.Table.Columns.Contains(TABLE.PhlebotomistId))
			{
				dtRow[TABLE.PhlebotomistId] =  this.Phlebotomist;
			}

			//				//SpecimenExpirationDate is only filled in on the first save
			//				//After that, the ExtendedExpirationDate is filled in
			//				if (this.IsNew)
			//				{
			//					if(dtRow.Table.Columns.Contains(TABLE.SpecimenExpirationDate))
			//					{
			//						dtRow[TABLE.SpecimenExpirationDate] =  this.SpecimenExpirationDate;
			//					}
			//				}
			//				else
			//				{
			//					if(dtRow.Table.Columns.Contains(TABLE.ExtendedExpirationDate))
			//					{
			//						dtRow[TABLE.ExtendedExpirationDate] =  this.SpecimenExpirationDate;
			//					}
			//				}


			if(dtRow.Table.Columns.Contains(TABLE.UnacceptableSpecimenIndicator))
			{
				dtRow[TABLE.UnacceptableSpecimenIndicator] =  this.UnacceptableSpecimenIndicator;
			}

			if (this.UnacceptableSpecimenIndicator && (this.UnacceptableSpecimenComment != null))
			{
				if(dtRow.Table.Columns.Contains(TABLE.CannedCommentGuid))
				{
					dtRow[TABLE.CannedCommentGuid] =  this.UnacceptableSpecimenComment.CannedCommentGuid;
				}

				if(dtRow.Table.Columns.Contains(TABLE.SpecimenComments))
				{
					dtRow[TABLE.SpecimenComments] =  this.UnacceptableSpecimenDetails;
				}
			}
			else
			{
				dtRow[TABLE.CannedCommentGuid] = DBNull.Value;
				dtRow[TABLE.SpecimenComments] =  DBNull.Value;
			}

			if(dtRow.Table.Columns.Contains(TABLE.DivisionCode))
			{
				dtRow[TABLE.DivisionCode] =  this.DivisionCode;
			}

			if(dtRow.Table.Columns.Contains(TABLE.SpecimenStatusCode))
			{
				dtRow[TABLE.SpecimenStatusCode] =  this.SpecimenStatusCode;
			}

			if(dtRow.Table.Columns.Contains(TABLE.EntryTechId))
			{
				dtRow[TABLE.EntryTechId] =  this.ProcessingTechId;
			}

			//				//BR 35.11
			//				if (this.IsNew)
			//				{
			//					dtRow[TABLE.VbecsSpecimenNumber] =  this.SpecimenAccessionNumber;
			//				}
			//
			//				if (this.VBECSSpecimenNumber == string.Empty)
			//				{
			//					dtRow[TABLE.VbecsSpecimenNumber] =  DBNull.Value;
			//				}
			//				else
			//				{
			//					dtRow[TABLE.VbecsSpecimenNumber] =  this.VBECSSpecimenNumber;
			//				}
			//
			if (dtRow.Table.Columns.Contains(TABLE.SpecimenExpirationDate))
			{
				dtRow[TABLE.SpecimenExpirationDate] = this.SpecimenExpirationDate;
			}
			if (dtRow.Table.Columns.Contains(TABLE.MaintainableSpecimenIndicator))
			{
				dtRow[TABLE.MaintainableSpecimenIndicator] = this.MaintainableSpecimenIndicator;
			}
			//				if (this.IsExtended)
			//				{
			//					dtRow[TABLE.SpecimenExpirationDate] = DBNull.Value;
			//					dtRow[TABLE.ExtendedExpirationDate] = this.SpecimenExpirationDate;
			//				}
			//				else
			//				{
			//					dtRow[TABLE.SpecimenExpirationDate] = this.SpecimenExpirationDate;
			//					dtRow[TABLE.ExtendedExpirationDate] = DBNull.Value;
			//				}
			if (dtRow.Table.Columns.Contains(TABLE.RowVersion))
			{
				dtRow[TABLE.RowVersion] = this.RowVersion;
			}
				
			return dtRow;
		}


		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="2765">
		///		<ExpectedInput>Valid specimenUid and patient SSN</ExpectedInput>
		///		<ExpectedOutput>Valid data table</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2766">
		///		<ExpectedInput>Invalid specimenUid and patient SSN</ExpectedInput>
		///		<ExpectedOutput>NoRecordsFoundException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets data table for Order search screen
		/// </summary>
		/// <param name="specimenUid"></param>
		/// <param name="patientSsn"></param>
		/// <param name="patientLastName"></param>
		/// <returns></returns>
		public static DataTable GetSpecimenSearchDetails(string specimenUid, string patientSsn, string patientLastName)
		{
			DataTable dtDetails = DAL.Specimen.GetSpecimenSearchDetails(specimenUid, patientSsn, patientLastName);
			
			if (dtDetails.Rows.Count == 0)
			{
				throw new BOL.NoRecordsFoundException();
			}
			
			return dtDetails;
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/16/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5430"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5431"> 
		///		<ExpectedInput>INvalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetSpecimenHistory
		/// </summary>
		/// <param name="specimenGuid"></param>
		/// <returns></returns>
		public static DataTable GetSpecimenHistory(Guid specimenGuid)
		{
			DataTable dtSpecimenHistory = DAL.Specimen.GetSpecimenHistory(specimenGuid);
	
			return dtSpecimenHistory;
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/16/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3560"> 
		///		<ExpectedInput>Valid specimenUid and patient SSN</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3561"> 
		///		<ExpectedInput>Invalid parameters</ExpectedInput>
		///		<ExpectedOutput>NoRecordsFoundException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets data table for Specimen search screen
		/// </summary>
		/// <param name="specimenUid"></param>
		/// <param name="patientSsn"></param>
		/// <param name="patientLastName"></param>
		/// <param name="expiresAfter"></param>
		/// <returns></returns>
		public static DataTable GetSpecimenList(string specimenUid, string patientSsn, string patientLastName, DateTime expiresAfter)
		{
			DataTable dtDetails = DAL.Specimen.GetSpecimenList(specimenUid, patientSsn, patientLastName, expiresAfter);
			
			if (dtDetails.Rows.Count == 0)
			{
				throw new BOL.NoRecordsFoundException();
			}
			
			return dtDetails;
		}

		///<Developers>
		///	<Developer>COE User</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/16/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3532"> 
		///		<ExpectedInput>specimenUId</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3533"> 
		///		<ExpectedInput>Invalid Uid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetSpecimen
		/// </summary>
		/// <param name="specimenUid"></param>
		/// <returns></returns>
		public static DataTable GetSpecimen(string specimenUid)
		{
			return DAL.Specimen.GetSpecimen(specimenUid);
		}


		///<Developers>
		///	<Developer>COE User</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/16/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3562"> 
		///		<ExpectedInput>strings</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3563"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Public method to compare the specimen collection, received and expiration dates
		/// Any error messages are returned for use in error providers, BR_35.01
		/// </summary>
		/// <param name="collectionDateError">Collection Date Error string</param>
		/// <param name="receivedDateError">Received Date Error string</param>
		/// <param name="expirationDateError">Expiration Date Error string</param>
		public void CompareDates(ref string collectionDateError, ref string receivedDateError, ref string expirationDateError)
		{
			System.DateTime currentDate = VBECSDateTime.GetDivisionCurrentDateTime();
			//
			if (Common.VBECSDateTime.CompareDates(this.SpecimenCollectionDate,currentDate)>0)
			{
				collectionDateError = Common.StrRes.InfoMsg.UC035.NonFutureSpecimenCollectionDate().ResString;
			}
			else
			{
				collectionDateError = string.Empty;
			}
			//
			if (Common.VBECSDateTime.CompareDates(this.SpecimenReceivedDate,this.SpecimenCollectionDate)<0)
			{
				receivedDateError = Common.StrRes.InfoMsg.UC035.SpecimenCollectionAfterReceived().ResString;
			}
			else if (Common.VBECSDateTime.CompareDates(this.SpecimenReceivedDate,currentDate)>0)
			{
				receivedDateError = Common.StrRes.InfoMsg.UC035.NonFutureSpecimenReceivedDate().ResString;
			}
			else
			{
				receivedDateError = string.Empty;
			}
			//
			if (Common.VBECSDateTime.CompareDates(this.SpecimenExpirationDate,this.SpecimenCollectionDate)<0)
			{
				expirationDateError = Common.StrRes.InfoMsg.UC035.SpecimenExpirationAfterCollection().ResString;
			}
			else if (Common.VBECSDateTime.CompareDates(this.SpecimenExpirationDate,this.SpecimenReceivedDate)<0)
			{
				expirationDateError = Common.StrRes.InfoMsg.UC035.SpecimenExpirationAfterReceived().ResString;
			}
			else if (this._isOverMaxDays)
			{
				expirationDateError = Common.StrRes.InfoMsg.UC035.SpecimenExpirationDateOutOfRange(this.MaximumSpecimenExpirationDays.ToString()).ResString;
			}
			else
			{
				expirationDateError = string.Empty;
			}

		} 

		#endregion

	}
}


